/*
 *	DECARI.CPP -- Data Decompression Routines
 * 
 *  Adapted by Noose from Original code by Haruhiko Okumura
 *
 *  Revisions:
 *  26/5/98     Corrected cyclical buffer routines to
 *              minimise memory overheads
 *  26/5/98     Fixed huge gaping error which lead to repeat
 *              creations of the class failing
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "decari.hpp"

/********** Bit I/O **********/

void Error(char *message)
{
	printf("\n%s\n", message); // Erm, not much use in Windoze
	exit(EXIT_FAILURE);
} // Error(...)

int LZAri::GetBit(void)  /* Get one bit (0 or 1) */
{
	
	if ((umask >>= 1) == 0) {
		charbuffer = getc(infile);  umask = 128;
	}
	return ((charbuffer & umask) != 0);
} // LZAri::GetBit()

/********** Arithmetic Compression **********/

/*  If you are not familiar with arithmetic compression, you should read
		I. E. Witten, R. M. Neal, and J. G. Cleary,
			Communications of the ACM, Vol. 30, pp. 520-540 (1987),
	from which much of this has been borrowed.  */

void LZAri::StartModel(void)  /* Initialize model */
{
	int ch, sym, i;
	
	sym_cum[N_CHAR] = 0;
	for (sym = N_CHAR; sym >= 1; sym--) {
		ch = sym - 1;
		char_to_sym[ch] = sym;  sym_to_char[sym] = ch;
		sym_freq[sym] = 1;
		sym_cum[sym - 1] = sym_cum[sym] + sym_freq[sym];
	}
	sym_freq[0] = 0;  /* sentinel (!= sym_freq[1]) */
	position_cum[N] = 0;
	for (i = N; i >= 1; i--)
		position_cum[i - 1] = position_cum[i] + 10000 / (i + 200);
			/* empirical distribution function (quite tentative) */
			/* Please devise a better mechanism! */
} // LZAri::StartModel()

void LZAri::UpdateModel(int sym)
{
	int i, c, ch_i, ch_sym;
	
	if (sym_cum[0] >= MAX_CUM) {
		c = 0;
		for (i = N_CHAR; i > 0; i--) {
			sym_cum[i] = c;
			c += (sym_freq[i] = (sym_freq[i] + 1) >> 1);
		}
		sym_cum[0] = c;
	}
	for (i = sym; sym_freq[i] == sym_freq[i - 1]; i--) ;
	if (i < sym) {
		ch_i = sym_to_char[i];    ch_sym = sym_to_char[sym];
		sym_to_char[i] = ch_sym;  sym_to_char[sym] = ch_i;
		char_to_sym[ch_i] = sym;  char_to_sym[ch_sym] = i;
	}
	sym_freq[i]++;
	while (--i >= 0) sym_cum[i]++;
} // LZAri::UpdateModel(...)

void LZAri::StartDecode(void)
{
	int i;

	for (i = 0; i < M + 2; i++)
		value = 2 * value + GetBit();
} // LZAri::StartDecode()

int LZAri::DecodeChar(void)
{
	int	 sym, ch;
	unsigned long int  range;
	
	range = high - low;
	sym = BinarySearchSym((unsigned int)
		(((value - low + 1) * sym_cum[0] - 1) / range));
	high = low + (range * sym_cum[sym - 1]) / sym_cum[0];
	low +=       (range * sym_cum[sym    ]) / sym_cum[0];
	for ( ; ; ) {
		if (low >= Q2) {
			value -= Q2;  low -= Q2;  high -= Q2;
		} else if (low >= Q1 && high <= Q3) {
			value -= Q1;  low -= Q1;  high -= Q1;
		} else if (high > Q2) break;
		low += low;  high += high;
		value = 2 * value + GetBit();
	}
	ch = sym_to_char[sym];
	UpdateModel(sym);
	return ch;
} // LZAri::DecodeChar()

int LZAri::DecodePosition(void)
{
	int position;
	unsigned long int  range;
	
	range = high - low;
	position = BinarySearchPos((unsigned int)
		(((value - low + 1) * position_cum[0] - 1) / range));
	high = low + (range * position_cum[position    ]) / position_cum[0];
	low +=       (range * position_cum[position + 1]) / position_cum[0];
	for ( ; ; ) {
		if (low >= Q2) {
			value -= Q2;  low -= Q2;  high -= Q2;
		} else if (low >= Q1 && high <= Q3) {
			value -= Q1;  low -= Q1;  high -= Q1;
		} else if (high > Q2) break;
		low += low;  high += high;
		value = 2 * value + GetBit();
	}
	return position;
} // LZAri::DecodePosition()

int LZAri::BinarySearchSym(unsigned int x)
	/* 1      if x >= sym_cum[1],
	   N_CHAR if sym_cum[N_CHAR] > x,
	   i such that sym_cum[i - 1] > x >= sym_cum[i] otherwise */
{
	int i, j, k;
	
	i = 1;  j = N_CHAR;
	while (i < j) {
		k = (i + j) / 2;
		if (sym_cum[k] > x) i = k + 1;  else j = k;
	}
	return i;
} // LZAri::BinarySearchSym(...)

int LZAri::BinarySearchPos(unsigned int x)
	/* 0 if x >= position_cum[1],
	   N - 1 if position_cum[N] > x,
	   i such that position_cum[i] > x >= position_cum[i + 1] otherwise */
{
	int i, j, k;
	
	i = 1;  j = N;
	while (i < j) {
		k = (i + j) / 2;
		if (position_cum[k] > x) i = k + 1;  else j = k;
	}
	return i - 1;
} // LZAri::BinarySearchPos(...)


void LZAri::PutChar(int c) {
    if (RBOffset==MAX_READ_BUFF)
        RBOffset=0;
    ReadBuffer[RBOffset++]=(char)c;
    ReadBuffer[RBOffset]='\0';
} // LZAri::PutChar(...);

/*
 * LZAri::Decode(void)
 *
 * Decodes a single block (or byte) from the input stream.
 *
 *  In: Nothing
 * Out: Nothing
 *
 *   Created: 25/5/98 Noose
 * Revisions:
 */
void LZAri::Decode(void)
{
	int  i, j, k, c;

	c = DecodeChar();
	if (c < 256) {
		PutChar(c);  text_buf[r++] = (char)c;
		r &= (N - 1);  count++;
	} else {
		i = (r - DecodePosition() - 1) & (N - 1);
		j = c - 255 + THRESHOLD;
		for (k = 0; k < j; k++) {
			c = text_buf[(i + k) & (N - 1)];
			PutChar(c);  text_buf[r++] = (char)c;
			r &= (N - 1);  count++;
		}
	}
	if (count > printcount)
		printcount += 1024;
} // LZAri::Decode()

/*
 * LZAri::eof()
 *
 * Returns whether the file is at it's end yet.
 *
 *  In: Nothing
 * Out: integer     -   Boolean Flag
 *
 *   Created: 25/5/98 Noose
 * Revisions:
 */
int LZAri::eof() {
    return _eofFlag;
} // LZAri::eof()

int LZAri::CompleteBuffer(int *ai,int *aIndex,char *Buffer) {
    int EndFound,i,Index;

    EndFound=FALSE;

    Index=*aIndex;
    i=*ai;

    while ( (i!=RBOffset) && (EndFound==FALSE) ) {
        if (i==MAX_READ_BUFF)
            i=0;
        Buffer[Index]=ReadBuffer[i];
        if (ReadBuffer[i]=='\x0a')
            EndFound=TRUE;
        Index++; i++;
    }

    *ai=i;
    *aIndex=Index;

    return EndFound;
}

/*
 * LZAri::Readln(char *Buffer)
 * 
 * Reads a single line from the compressed source file.
 * The routine creates a cyclical buffer which holds any chars
 * which have been "read-ahead" because of the compression.
 * The cyclical buffer should be larger than the length of the
 * longest line to prevent overlap.
 *
 *  In: Buffer[]    -   Where to store the line data
 * Out: Nothing
 *
 *   Created: 25/5/98 Noose
 * Revisions:
 *            29/5/98 Noose - Fixed the stupid CompleteBuffer function
 */


void LZAri::Readln(char *Buffer) {
    int Index,i,EndFound;
    
    ReadBuffer[MAX_READ_BUFF]='\0'; // Ensures ReadBuffer[MAX_READ_BUFF]!=10

    Index=0;                    // Output buffer index
    EndFound=FALSE;             // Not found the end yet
    i=RBLineStart;              // Point to the start of the current line

    while ( (count<textsize) && (EndFound==FALSE) ) {
        if (i==RBOffset)        // Have we run out of data in the buffer?
            Decode();           // Decode another block
        if (count<textsize)     // Are we still before the end of file?
            EndFound=CompleteBuffer(&i,&Index,Buffer);
        else
            _eofFlag=TRUE;      // It's the end of the file
    }

    if (_eofFlag) 
        CompleteBuffer(&i,&Index,Buffer);

    RBLineStart=i;          // No more text in cyclical buffer
    Buffer[Index]='\0';         // Terminate string
} // LZAri::Readln(...)

/*
 * LZAri::LZAri(char *fn)
 *
 * Constructor function for the LZAri class.
 * Initialises all the variables for the decompression and
 * opens the input file specified by fn.
 *
 *  In: fn          -   FileName to load
 * Out: Nothing
 *
 *   Created: 25/5/98 Noose
 * Revisions:
 */
LZAri::LZAri(char *fn) {
    int i;
    char Buffer[5];

    umask=0;
    textsize = 0; codesize = 0; printcount = 0;

    low = 0; high = Q4; value = 0;
    shifts = 0;  // counts for magnifying low and high around Q2 

    infile=fopen(fn,"rb");
    fread(Buffer,3,1,infile);
    Buffer[3]='\0';
    if (strcmp(Buffer,"SCR"))
        Error("Not a valid SCR file.");

	if (fread(&textsize, sizeof textsize, 1, infile) < 1)
		Error("Read Error");  /* read size of text */
	if (textsize == 0) return;
	StartDecode();  StartModel();
	for (i = 0; i < N - F; i++) text_buf[i] = ' ';
	r = N - F;
    RBOffset=0;
    RBLineStart=0;
    count=0;
    for (i=0;i<MAX_READ_BUFF;i++)
        ReadBuffer[i]='\0';
    _eofFlag=FALSE;
} // LZAri::LZAri(...)

/*
 * LZAri::~LZAri()
 *
 * Deconstuctor function for the LZAri Class.
 * Closes the open file.
 */
LZAri::~LZAri() {
    fclose(infile);             // Close the input file
} // LZAri::~LZAri()

